------------------------------------------------------------------------------------------------------------------------
-- INVENTURERFASSUNG
------------------------------------------------------------------------------------------------------------------------

--  api.inventur_sync_script()
CREATE OR REPLACE FUNCTION api.inventur_sync_script()
RETURNS text
AS $$
SELECT $html$
<script>
async function syncOfflineDaten() {
  const loader = document.getElementById('sync-loader');
  if (loader) loader.style.display = 'inline-block';

  const daten = JSON.parse(localStorage.getItem('offlineInventur') || '[]');
  if (daten.length === 0) {
    if (loader) loader.style.display = 'none';
    return;
  }

  try {
    const ping = await fetch("./ping", { method: "HEAD" });
    if (!ping.ok) throw new Error("Ping fehlgeschlagen");
  } catch (e) {
    console.warn("❌ Sync abgebrochen: Server nicht erreichbar", e);
    if (loader) loader.style.display = 'none';
    return;
  }

  const verbleibend = [];

  for (const eintrag of daten) {
    try {
      const res = await fetch('./inventur_zaehlliste_artikel_erfassen', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'Authorization': 'Bearer ' + getCookie('jwt')
        },
        body: JSON.stringify({
          p_il_id: eintrag.id,
          p_anzahl: eintrag.ist,
          p_kommentar: eintrag.kommentar
        })
      });

      const contentType = res.headers.get("Content-Type") || "";
      if (res.ok && contentType.includes("application/json")) {
        const json = await res.json();
        console.log('✅ Synchronisiert:', json);

        const row = document.querySelector(`th[id="${eintrag.id}"]`)?.parentElement;
        if (row) {
          row.querySelectorAll("td")[2].textContent = eintrag.ist || '';
        }
        bewerteAbweichung();
      } else {
        throw new Error("Antwort nicht OK oder kein JSON: " + contentType);
      }
    } catch (err) {
      console.warn('❌ Fehler bei Sync, Eintrag bleibt lokal:', eintrag);
      verbleibend.push(eintrag);
    }
  }

  localStorage.setItem('offlineInventur', JSON.stringify(verbleibend));
  aktualisiereOfflineStatus();
  if (loader) loader.style.display = 'none';
}

function getCookie(name) {
  const match = document.cookie.match(new RegExp('(^| )' + name + '=([^;]+)'));
  return match ? match[2] : null;
}

window.addEventListener('online', () => {
  console.log("🌐 Wieder online – versuche Synchronisation…");
  syncOfflineDaten();
});

document.addEventListener("DOMContentLoaded", () => {
  const form = document.getElementById("inventur-form");
  const submitBtn = form?.querySelector('button[type="submit"]');

  if (form && submitBtn) {
    submitBtn.addEventListener("click", async function (e) {
      e.preventDefault();

      const daten = {
        id: form.querySelector('[name="p_il_id"]').value,
        ist: form.querySelector('[name="p_anzahl"]').value,
        kommentar: form.querySelector('[name="p_kommentar"]').value,
        timestamp: Date.now()
      };

      try {
        const ping = await fetch("./ping", { method: "HEAD" });
        if (!ping.ok) throw new Error("Serverantwort nicht OK");
      } catch (pingError) {
        console.warn("📴 Server nicht erreichbar – speichere lokal:", pingError);

        const offline = JSON.parse(localStorage.getItem('offlineInventur') || '[]');
        offline.push(daten);
        localStorage.setItem('offlineInventur', JSON.stringify(offline));

        const row = document.querySelector(`th[id="${daten.id}"]`)?.parentElement;
        if (row) {
          row.querySelectorAll("td")[2].textContent = daten.ist || '';
        }

        alert("📦 Zählung wurde offline gespeichert.");
        document.getElementById('erfassung-modal')?.close();
        bewerteAbweichung();
        aktualisiereOfflineStatus();
        return;
      }

      try {
        const res = await fetch('./inventur_zaehlliste_artikel_erfassen', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'Authorization': 'Bearer ' + getCookie('jwt')
          },
          body: JSON.stringify({
            p_il_id: daten.id,
            p_anzahl: daten.ist,
            p_kommentar: daten.kommentar
          })
        });

        if (res.ok) {
          const json = await res.json();
          console.log("✅ Online gespeichert", json);

          const row = document.querySelector(`th[id="${daten.id}"]`)?.parentElement;
          if (row) {
            row.querySelectorAll("td")[2].textContent = daten.ist || '';
          }

          document.getElementById('erfassung-modal')?.close();
          bewerteAbweichung();
        } else {
          const text = await res.text();
          throw new Error("Serverfehler: " + res.status + " – " + text);
        }
      } catch (err) {
        alert("❌ Fehler beim Speichern online. Bitte später erneut versuchen.");
        console.error(err);
        document.getElementById('erfassung-modal')?.close();
      }
    });
  }

  setInterval(() => {
    const daten = JSON.parse(localStorage.getItem('offlineInventur') || '[]');
    if (daten.length > 0) {
      console.log("⏳ Automatischer Sync-Versuch...");
      syncOfflineDaten();
    }
  }, 30000);

  aktualisiereOfflineStatus();
});

function zeigeOfflineHinweis() {
  const daten = JSON.parse(localStorage.getItem('offlineInventur') || '[]');
  const meldung = document.getElementById('meldung');
  if (daten.length > 0 && meldung) {
    meldung.innerHTML = `<p style=\"color: orange;\">⚠️ ${daten.length} Zählung(en) sind offline gespeichert.</p>`;
  }
}

function aktualisiereOfflineStatus() {
  const daten = JSON.parse(localStorage.getItem('offlineInventur') || '[]');
  const meldung = document.getElementById('meldung');
  if (!meldung) return;

  meldung.innerHTML = '';
  if (daten.length > 0) {
    meldung.innerHTML = `<p style=\"color: orange; display: inline-block;\">⚠️ ${daten.length} Zählung(en) sind offline gespeichert.</p>
    <span id=\"sync-loader\" class=\"loading\" style=\"margin-left: 0.5rem; display: none;\"></span>`;

    const btn = document.createElement('button');
    btn.textContent = 'Jetzt synchronisieren';
    btn.className = 'secondary';
    btn.style.marginLeft = '1rem';
    btn.onclick = () => {
      console.log("🔘 Manueller Sync gestartet");
      syncOfflineDaten();
    };
    meldung.appendChild(btn);
  }
}

document.addEventListener('DOMContentLoaded', zeigeOfflineHinweis);
</script>
$html$;
$$ LANGUAGE sql IMMUTABLE;


-- Inventur Übersichtsseite
--- Funktion generiert HTML-Seite mit einer Übersicht der offenen Inventuren
CREATE OR REPLACE FUNCTION api.inventur()
  RETURNS "text/html"
  AS $$
  SELECT concat_ws( E'\n',
  '<!DOCTYPE html>', '<html lang="de">',
  api.html_head_get(),
  $html$
  <body class="bg-gray-50 text-gray-800">
  <main class="max-w-5xl mx-auto px-4 py-6">

    <!-- Navigation -->
    <nav class="flex justify-between items-center mb-6">
      <img src="./file?id=prodat_logo" alt="Prodat Logo" class="h-32" />
      <button id="loginBtn" class="hidden px-4 py-2 border border-gray-400 rounded hover:bg-gray-100">Login</button>
    </nav>

    <!-- Haupttitel -->
    <h1 class="text-2xl font-semibold mb-4">Offene Inventuren</h1>

    <!-- Toolbar -->
    <div class="flex gap-2 items-center mb-6">
      <input type="search" name="filter" placeholder="Filtern nach Name oder Datum"
        class="flex-1 px-3 py-2 border rounded"
        hx-get="./inventur_liste"
        hx-trigger="keyup changed delay:600ms"
        hx-target="#inventur-liste"
        hx-params="filter"
        hx-headers='{"Accept": "text/html"}'
        hx-swap="innerHTML" />

      <button
        class="px-4 py-2 bg-gray-100 border rounded hover:bg-gray-200"
        title="Neu laden"
        hx-get="./inventur_liste"
        hx-target="#inventur-liste"
        hx-headers='{"Accept": "text/html"}'
        hx-swap="innerHTML">
        Alle
      </button>
    </div>

    <!-- Liste der Inventuren -->
    <div id="inventur-liste" class="grid gap-4"
      hx-get="./inventur_liste"
      hx-trigger="load"
      hx-target="this"
      hx-headers='{"Accept": "text/html"}'
      hx-swap="innerHTML">
      <p>⏳ Lade Inventuren…</p>
    </div>
  </main>
  <script>
  document.getElementById("loginBtn").addEventListener("click", function () {
    // Umleitung zur Login-Seite
    window.location.href = './login_page';
  });
  </script>
  </body>
  </html>$html$ );
  $$ LANGUAGE sql IMMUTABLE;


-- Funktion generiert die einzelnen Karten zu offenen Inventuren in der Übersichtsseite
CREATE OR REPLACE FUNCTION api.inventur_liste(
    filter text DEFAULT null
  )
  RETURNS "text/html"
  AS $$
  DECLARE
    iv     record;
    result text := '';
  BEGIN
    FOR iv IN
      SELECT iv_nr, iv_bdat, count(il_id) AS il_all, count(il_id) FILTER (WHERE il_zaedat IS NOT null) AS il_done
      FROM inv
      LEFT JOIN invlag ON il_iv_nr = iv_nr
      WHERE iv_edat IS NULL
        AND (
          filter IS NULL OR
          iv_nr ILIKE '%' || filter || '%' OR
          to_char(iv_bdat, 'DD.MM.YYYY') ILIKE '%' || filter || '%'
        )
      GROUP BY iv_nr, iv_bdat
      ORDER BY iv_bdat, iv_nr
    LOOP
      result := result || format(
        $article$<article class="cursor-pointer p-4 rounded-lg bg-white shadow hover:bg-blue-50 transition active:scale-95"
              onclick="location.href='./inventur_zaehlliste?iv_nr=%s'">
            <h2 class="text-lg font-semibold">Inventur - %s</h2>
            <p class="text-sm text-gray-600">📅 Eröffnet am: %s</p>
            <progress value=%s max=%s class="w-full mt-2" />
        </article>$article$,
        iv.iv_nr,
        iv.iv_nr,
        to_char(iv.iv_bdat, 'DD.MM.YYYY'),
        iv.il_done, iv.il_all
      );
    END LOOP;

    IF result = '' THEN
      result := '<p>ℹ️ Keine passenden offenen Inventuren gefunden.</p>';
    END IF;

    RETURN result;
  END;
  $$ LANGUAGE plpgsql;


-- Funktion generiert HTML-Seite für die Zählliste zu einer bestimmten Inventur
--DROP FUNCTION IF EXISTS api.inventur_zaehlliste(character varying);
CREATE OR REPLACE FUNCTION api.inventur_zaehlliste(iv_nr varchar)
  RETURNS "text/html"
  SECURITY DEFINER
  AS $BODY$
SELECT concat_ws(E'\n',
'<!DOCTYPE html>', '<html lang="de">',
api.html_head_get('Prodat ERP - Zählliste'),
$html$
<body class="bg-gray-50 text-gray-900 min-h-screen">
  <main class="max-w-3xl mx-auto py-8 px-4">
    <h1 class="text-2xl font-bold flex items-center gap-2 mb-6">📋 Zählliste - $html$ || iv_nr || $html$</h1>

    <form onsubmit="return false;" class="flex gap-2 items-center mb-4">
      <input
        type="search"
        id="artikel-suche"
        name="artikel-suche"
        placeholder="Filtern nach Artikelnummer oder Bezeichnung"
        aria-label="Artikelsuche"
        autocomplete="off"
        class="flex-1 px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-400 transition"
      />
      <button
        type="button"
        id="scanner-button"
        aria-label="Barcode scannen"
        class="px-3 py-2 bg-gray-100 hover:bg-blue-100 text-lg rounded-lg border border-gray-300 shadow-sm transition"
      >
        📷
      </button>
    </form>
    <video id="scanner-preview" class="w-full max-h-[200px] hidden mb-4 rounded-lg shadow" style="display: none;"></video>

    <div id="meldung" class="my-2 text-sm text-blue-600"></div>

    <div id="liste" class="overflow-x-auto rounded-lg shadow bg-white">
      <table class="min-w-full table-auto text-left">
        <thead>
          <tr class="bg-gray-100">
            <th class="px-4 py-2 font-semibold">Artikelnummer</th>
            <th class="px-4 py-2 font-semibold">Artikelbezeichnung</th>
            <th class="px-4 py-2 font-semibold">Inventurbestand</th>
            <th class="px-4 py-2 font-semibold">Zählbestand</th>
          </tr>
        </thead>
        <tbody id="zaehlliste"
          hx-get="./inventur_zaehlliste_artikel?iv_nr=$html$ || iv_nr || $html$"
          hx-trigger="load"
          hx-target="this"
          hx-headers='{"Accept": "text/html"}'
          hx-swap="innerHTML">
          <tr><td colspan="4" class="px-4 py-2 text-center">⏳ Lade Zählliste…</td></tr>
        </tbody>
      </table>
    </div>

    <dialog id="erfassung-modal" class="rounded-lg shadow-xl max-w-md w-full mx-auto p-0">
      <article class="p-6 bg-white rounded-lg">
        <header class="flex items-center justify-between mb-4">
          <button aria-label="Close" rel="prev" onclick="closeModal()" class="text-gray-400 hover:text-gray-700 text-xl font-bold">&times;</button>
          <div>
            <h2 id="modal-artikelnummer" class="text-lg font-semibold">Artikelnummer</h2>
            <h3 id="modal-artikelname" class="text-md text-gray-500">Artikelname</h3>
          </div>
        </header>
        <form id="inventur-form"
          hx-post="./inventur_zaehlliste_artikel_erfassen"
          hx-include="[name=p_il_id],[name=p_anzahl],[name=p_kommentar]"
          hx-params="not soll_bestand"
          hx-trigger="submit"
          hx-swap="none"
          class="space-y-4"
        >
          <input type="hidden" name="p_il_id">
          <div>
            <label for="soll_bestand" class="block text-sm font-medium mb-1">Soll-Bestand (Inventurmenge):</label>
            <input type="number" name="soll_bestand" readonly class="w-full px-3 py-2 border rounded-lg bg-gray-100" />
          </div>
          <div>
            <label for="p_anzahl" class="block text-sm font-medium mb-1">Ist-Bestand (Zählmenge):</label>
            <input type="number" name="p_anzahl" required class="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-400" />
          </div>
          <div>
            <label for="p_kommentar" class="block text-sm font-medium mb-1">Kommentar:</label>
            <textarea name="p_kommentar" class="w-full px-3 py-2 border rounded-lg"></textarea>
          </div>
          <footer class="flex gap-2 mt-4">
            <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 font-semibold">Speichern</button>
            <button type="button" class="px-4 py-2 bg-gray-200 hover:bg-gray-300 rounded-lg text-gray-700 font-semibold" onclick="closeModal()">Abbrechen</button>
          </footer>
        </form>
      </article>
    </dialog>
  </main>

<script>
function setupRowClickHandlers() {
  const rows = document.querySelectorAll('#liste tbody tr');
  rows.forEach(row => {
    row.style.cursor = 'pointer';

    // Doppelte Bindung vermeiden
    row.removeEventListener('click', row._clickHandler);

    const handler = function () {
      const invlagIdElement = row.querySelector('th');
      const invlagId = invlagIdElement ? invlagIdElement.id : '';
      const artikelNummer = invlagIdElement ? invlagIdElement.textContent : '';

      const artikelNameElement = row.querySelector('td');
      const artikelName = artikelNameElement ? artikelNameElement.textContent : '';

      document.querySelector('input[name="p_il_id"]').value = invlagId;
      document.getElementById('modal-artikelnummer').textContent = artikelNummer;
      document.getElementById('modal-artikelname').textContent = artikelName;

      const inventurBestandElement = row.querySelectorAll('td')[1];
      const inventurBestand = inventurBestandElement ? inventurBestandElement.textContent : '';
      document.querySelector('input[name="soll_bestand"]').value = inventurBestand;

      const zaehlBestandElement = row.querySelectorAll('td')[2];
      const zaehlBestand = zaehlBestandElement ? zaehlBestandElement.textContent : '';
      document.querySelector('input[name="p_anzahl"]').value = zaehlBestand;

      document.getElementById('erfassung-modal').showModal();
    };

    // Event speichern und binden
    row._clickHandler = handler;
    row.addEventListener('click', handler);
  });
}

function closeModal() {
  document.getElementById('erfassung-modal').close();
}

function bewerteAbweichung() {
  document.querySelectorAll("#liste tbody tr").forEach(row => {
    // Tailwind-Hintergrundklassen entfernen
    row.classList.remove(
      "bg-green-50", "bg-red-50", "bg-yellow-50",
      "text-green-800", "text-red-800", "text-yellow-800",
      "font-semibold", "italic"
    );

    const tds = row.querySelectorAll("td");
    if (tds.length >= 3) {
      const soll = parseFloat(tds[1].textContent);
      const ist = parseFloat(tds[2].textContent);

      if (isNaN(ist)) {
        // Noch nicht erfasst: gelb, kursiv
        row.classList.add("bg-yellow-50", "text-yellow-800", "italic");
      } else if (!isNaN(soll) && soll > 0) {
        const abw = Math.abs(ist - soll) / soll;
        if (abw < 0.1) {
          // OK: grün hinterlegt
          row.classList.add("bg-green-50", "text-green-800", "font-semibold");
        } else {
          // Grosse Abweichung: rot hinterlegt
          row.classList.add("bg-red-50", "text-red-800", "font-semibold");
        }
      } else if (soll === 0 && ist > 0) {
        // Bestand sollte null sein, ist aber vorhanden
        row.classList.add("bg-red-50", "text-red-800", "font-semibold");
      } else if (soll === 0 && ist === 0) {
        row.classList.add("bg-green-50", "text-green-800", "font-semibold");
      }
    }
  });
}


// Initiale Bindung beim Laden
document.addEventListener('DOMContentLoaded', () => {
  setupRowClickHandlers();
  bewerteAbweichung();
});

// HTMX: Nach DOM-Update (z. B. nach hx-swap)
document.body.addEventListener('htmx:afterSettle', () => {
  setupRowClickHandlers();
  bewerteAbweichung();
});

// Suche
document.addEventListener('DOMContentLoaded', () => {
  const suchfeld = document.getElementById('artikel-suche');
  if (suchfeld) {
    suchfeld.addEventListener('input', () => {
      const suchbegriff = suchfeld.value.toLowerCase();
      document.querySelectorAll("#zaehlliste tr").forEach(row => {
        const text = row.innerText.toLowerCase();
        row.style.display = text.includes(suchbegriff) ? '' : 'none';
        const sichtbare = [...document.querySelectorAll("#zaehlliste tr")]
          .filter(row => row.style.display !== 'none');
        document.getElementById("meldung").textContent = sichtbare.length + " Einträge gefunden";
      });
    });
  }
});

// Scanner
const suchfeld = document.getElementById('artikel-suche');
const scannerButton = document.getElementById('scanner-button');
const preview = document.getElementById('scanner-preview');

let scannerAktiv = false;

function startScanner() {
  Quagga.init({
    inputStream: {
      name: "Live",
      type: "LiveStream",
      target: preview,
      constraints: {
        facingMode: "environment"
      }
    },
    decoder: {
      readers: ["code_128_reader", "ean_reader", "ean_8_reader"]
    }
  }, err => {
    if (err) {
      console.error(err);
      return;
    }
    Quagga.start();
    preview.style.display = 'block';
    scannerButton.textContent = '🛑'; // Stop-Symbol
    scannerAktiv = true;
  });

  Quagga.onDetected(result => {
    const code = result.codeResult.code;
    stopScanner();
    suchfeld.value = code;
    suchfeld.dispatchEvent(new Event('input')); // Filter auslösen
  });
}

function stopScanner() {
  if (scannerAktiv) {
    Quagga.stop();
    Quagga.offDetected(); // Wichtig: sonst mehrfach ausgelöst
    preview.style.display = 'none';
    scannerButton.textContent = '📷';
    scannerAktiv = false;
  }
}

scannerButton.addEventListener('click', () => {
  if (!scannerAktiv) {
    startScanner();
  } else {
    stopScanner();
  }
});

</script>$html$,
api.inventur_sync_script(),
$html$
</body>
</html>
$html$ )
$BODY$ LANGUAGE sql;


-- Funktion generiert Einträge der Zähliste für eine bestimmte Inventur
--DROP FUNCTION IF EXISTS api.inventur_zaehlliste_artikel(character varying);
CREATE OR REPLACE FUNCTION api.inventur_zaehlliste_artikel(iv_nr varchar)
  RETURNS "text/html"
  SECURITY DEFINER
  AS $$
    SELECT
      string_agg(
        format(
          '<tr class="hover:bg-blue-50 cursor-pointer transition">
            <th scope="row" id="%s" class="px-4 py-2 font-medium text-gray-900">%s</th>
            <td class="px-4 py-2 text-gray-700">%s</td>
            <td class="px-4 py-2 text-right text-gray-700">%s</td>
            <td class="px-4 py-2 text-right text-gray-700">%s</td>
          </tr>',
          il_id, il_aknr,
          ak_bez,
          COALESCE(il_anztot::text, ''),
          CASE WHEN il_zaedat IS NULL THEN '' ELSE COALESCE(il_anzist::text, '') END
        ),
        ''
      )
    FROM invlag
    JOIN art ON il_aknr = ak_nr
    WHERE il_iv_nr = iv_nr;
  $$ LANGUAGE sql STABLE;

-- Funktion schreibt gezählte Werte in Zählliste zurück
CREATE OR REPLACE FUNCTION api.inventur_zaehlliste_artikel_erfassen(
    p_il_id     integer,
    p_anzahl    numeric,
    p_kommentar text DEFAULT NULL
)
RETURNS json
SECURITY DEFINER
AS $$
DECLARE
  r record;
BEGIN
  UPDATE invlag
     SET il_anzist = p_anzahl,
         il_txt = p_kommentar,
         il_zaedat = now()
   WHERE il_id = p_il_id
   RETURNING il_id, il_anzist, il_txt INTO r;

  RETURN json_build_object(
    'id', r.il_id,
    'ist', r.il_anzist,
    'kommentar', r.il_txt
  );
END;
$$ LANGUAGE plpgsql;